/*
 * BQ27x00 battery driver
 *
 * Copyright (C) 2008 Rodolfo Giometti <giometti@linux.it>
 * Copyright (C) 2008 Eurotech S.p.A. <info@eurotech.it>
 *
 * Based on a previous work by Copyright (C) 2008 Texas Instruments, Inc.
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
#include <linux/module.h>
#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/idr.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <asm/unaligned.h>

#define DRIVER_VERSION			"1.1.0"

#define BQ27x00_REG_TEMP		0x02
#define BQ27x00_REG_VOLT		0x04
#define BQ27x00_REG_AI			0x10
#define BQ27x00_REG_FLAGS		0x06
#define BQ27x00_REG_TTE			0x16
#define BQ27x00_REG_TTF			0x18
#define BQ27x00_REG_TTECP		0x26

#define BQ27000_REG_RSOC		0x0B /* Relative State-of-Charge */
#define BQ27000_FLAG_CHGS		BIT(7)

#define BQ27500_REG_SOC			0x1c
#define BQ27500_FLAG_DSC		BIT(0)
#define BQ27500_FLAG_FC			BIT(9)

#define BQ27410_DELAY			1000
/* If the system has several batteries we need a different name for each
 * of them...
 */
static DEFINE_IDR(battery_id);
static DEFINE_MUTEX(battery_mutex);

struct bq27x00_device_info;
struct bq27x00_access_methods {
	int (*read)(u8 reg, int *rt_value, int b_single,
		struct bq27x00_device_info *di);
	int (*write)(u8 reg, u8 *wr_value, int count,
		struct bq27x00_device_info *di);
};

enum bq27x00_chip { BQ27000, BQ27500 };

struct bq27x00_device_info {
	struct device 		*dev;
	int			id;
	struct bq27x00_access_methods	*bus;
	struct power_supply	bat;
	enum bq27x00_chip	chip;

	struct i2c_client	*client;
	struct delayed_work	work;
};

static enum power_supply_property bq27x00_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CURRENT_NOW,
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_TEMP,
//	POWER_SUPPLY_PROP_TIME_TO_EMPTY_NOW,
//	POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG,
//	POWER_SUPPLY_PROP_TIME_TO_FULL_NOW,
};

/*
 * Common code for BQ27x00 devices
 */

static int bq27x00_read(u8 reg, int *rt_value, int b_single,
			struct bq27x00_device_info *di)
{
	return di->bus->read(reg, rt_value, b_single, di);
}

/*
 * Return the battery temperature in tenths of degree Celsius
 * Or < 0 if something fails.
 */
static int bq27x00_battery_temperature(struct bq27x00_device_info *di)
{
	int ret;
	int temp = 0;

	ret = bq27x00_read(BQ27x00_REG_TEMP, &temp, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading temperature\n");
		return ret;
	}

	if (di->chip == BQ27500)
		return temp - 2731;
	else
		return ((temp >> 2) - 273) * 10;
}

/*
 * Return the battery Voltage in milivolts
 * Or < 0 if something fails.
 */
static int bq27x00_battery_voltage(struct bq27x00_device_info *di)
{
	int ret;
	int volt = 0;

	ret = bq27x00_read(BQ27x00_REG_VOLT, &volt, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading voltage\n");
		return ret;
	}

	return volt * 1000;
}

/*
 * Return the battery average current
 * Note that current can be negative signed as well
 * Or 0 if something fails.
 */
static int bq27x00_battery_current(struct bq27x00_device_info *di)
{
	int ret;
	int curr = 0;
	int flags = 0;

	ret = bq27x00_read(BQ27x00_REG_AI, &curr, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading current\n");
		return 0;
	}

	if (di->chip == BQ27500) {
		/* bq27500 returns signed value */
		curr = (int)(s16)curr;
	} else {
		ret = bq27x00_read(BQ27x00_REG_FLAGS, &flags, 0, di);
		if (ret < 0) {
			dev_err(di->dev, "error reading flags\n");
			return 0;
		}
		if (flags & BQ27000_FLAG_CHGS) {
			dev_dbg(di->dev, "negative current!\n");
			curr = -curr;
		}
	}

	return curr * 1000;
}

/*
 * Return the battery Relative State-of-Charge
 * Or < 0 if something fails.
 */
static int bq27x00_battery_rsoc(struct bq27x00_device_info *di)
{
	int ret;
	int rsoc = 0;

	if (di->chip == BQ27500)
		ret = bq27x00_read(BQ27500_REG_SOC, &rsoc, 0, di);
	else
		ret = bq27x00_read(BQ27000_REG_RSOC, &rsoc, 1, di);
	if (ret) {
		dev_err(di->dev, "error reading relative State-of-Charge\n");
		return ret;
	}

	return rsoc;
}

static int bq27x00_battery_status(struct bq27x00_device_info *di,
				  union power_supply_propval *val)
{
	int flags = 0;
	int status;
	int ret;

	ret = bq27x00_read(BQ27x00_REG_FLAGS, &flags, 0, di);
	if (ret < 0) {
		dev_err(di->dev, "error reading flags\n");
		return ret;
	}

	if (di->chip == BQ27500) {
		if (flags & BQ27500_FLAG_FC)
			status = POWER_SUPPLY_STATUS_FULL;
		else if (flags & BQ27500_FLAG_DSC)
			status = POWER_SUPPLY_STATUS_DISCHARGING;
		else
			status = POWER_SUPPLY_STATUS_CHARGING;
	} else {
		if (flags & BQ27000_FLAG_CHGS)
			status = POWER_SUPPLY_STATUS_CHARGING;
		else
			status = POWER_SUPPLY_STATUS_DISCHARGING;
	}

	val->intval = status;
	return 0;
}

/*
 * Read a time register.
 * Return < 0 if something fails.
 */
static int bq27x00_battery_time(struct bq27x00_device_info *di, int reg,
				union power_supply_propval *val)
{
	int tval = 0;
	int ret;

	ret = bq27x00_read(reg, &tval, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading register %02x\n", reg);
		return ret;
	}

	if (tval == 65535)
		return -ENODATA;

	val->intval = tval * 60;
	return 0;
}

#define to_bq27x00_device_info(x) container_of((x), \
				struct bq27x00_device_info, bat);

static int bq27x00_battery_get_property(struct power_supply *psy,
					enum power_supply_property psp,
					union power_supply_propval *val)
{
	int ret = 0;
	struct bq27x00_device_info *di = to_bq27x00_device_info(psy);
	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		ret = bq27x00_battery_status(di, val);
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = bq27x00_battery_voltage(di);
		if (psp == POWER_SUPPLY_PROP_PRESENT)
			val->intval = val->intval <= 0 ? 0 : 1;
		break;
	case POWER_SUPPLY_PROP_CURRENT_NOW:
		val->intval = bq27x00_battery_current(di);
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		val->intval = bq27x00_battery_rsoc(di);
		break;
	case POWER_SUPPLY_PROP_TEMP:
		val->intval = bq27x00_battery_temperature(di);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_NOW:
		ret = bq27x00_battery_time(di, BQ27x00_REG_TTE, val);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG:
		ret = bq27x00_battery_time(di, BQ27x00_REG_TTECP, val);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_FULL_NOW:
		ret = bq27x00_battery_time(di, BQ27x00_REG_TTF, val);
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

static void bq27x00_powersupply_init(struct bq27x00_device_info *di)
{
	di->bat.type = POWER_SUPPLY_TYPE_BATTERY;
	di->bat.properties = bq27x00_battery_props;
	di->bat.num_properties = ARRAY_SIZE(bq27x00_battery_props);
	di->bat.get_property = bq27x00_battery_get_property;
	di->bat.external_power_changed = NULL;
}

/*
 * i2c specific code
 */

static int bq27x00_read_i2c(u8 reg, int *rt_value, int b_single,
			struct bq27x00_device_info *di)
{
	struct i2c_client *client = di->client;
	struct i2c_msg msg[1];
	unsigned char data[2];
	int err;

	if (!client->adapter)
		return -ENODEV;

	msg->addr = client->addr;
	msg->flags = 0;
	msg->len = 1;
	msg->buf = data;

	data[0] = reg;
	err = i2c_transfer(client->adapter, msg, 1);

	if (err >= 0) {
		if (!b_single)
			msg->len = 2;
		else
			msg->len = 1;

		msg->flags = I2C_M_RD;
		err = i2c_transfer(client->adapter, msg, 1);
		if (err >= 0) {
			if (!b_single)
				*rt_value = get_unaligned_le16(data);
			else
				*rt_value = data[0];

			return 0;
		}
	}
	return err;
}

u8 bq27410_CheckSum(unsigned char* buf,int cnt)
{
	int sum=0;
	int i;
	for(i=0;i<cnt;i++){
		sum = sum + buf[i];
	}
	sum = sum & 0xFF;
	sum = 255-sum;
	return (u8)sum;
}

static int bq27410_battery_config(struct bq27x00_device_info *di)
{
	int ret;
	u8 buf[32];
	u8 checksum=0;

	struct i2c_client *client = di->client;

	//set battery temperature
	ret=i2c_smbus_write_word_data(client, 0x02, 0x0b9d);
	if(ret!=0)
		return ret;
	//battery insert
	ret=i2c_smbus_write_word_data(client, 0x00, 0x000c);
	if(ret!=0)
		return ret;

	ret=i2c_smbus_read_word_data(client,0x3c);
	if(ret == 0x960)//if design capacity == 2400mAh, it has been configed
		return 0;

	//access data flash set design capacity
	//blockdatacontrol = 0
	ret=i2c_smbus_write_byte_data(client, 0x61, 0);
	if(ret!=0)
		return ret;
	//set dataflashclass(get design capacity)
	ret=i2c_smbus_write_byte_data(client, 0x3e, 48);
	if(ret!=0)
		return ret;
	//set dataflashblock
	ret=i2c_smbus_write_byte_data(client, 0x3f, 0);
	if(ret!=0)
		return ret;
	//read blockdata
	ret=i2c_smbus_read_i2c_block_data(client,0x40,32,buf);
	if(ret<0)
		return ret;
	//write blockdata space design capacity
	ret=i2c_smbus_write_word_data(client, 0x53, 0x6009);
	if(ret!=0)
		return ret;
	//read blockdata
	ret=i2c_smbus_read_i2c_block_data(client,0x40,32,buf);
	if(ret<0)
		return ret;
	//blockdata checksum
	checksum=bq27410_CheckSum(buf,32);
	//write checksum to blockdatachecksum
	ret=i2c_smbus_write_byte_data(client, 0x60, checksum);
	if(ret!=0)
		return ret;

	ret=i2c_smbus_read_word_data(client,0x00);
	printk("control: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x02);
	printk("temperature: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x04);
	printk("voltage: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x06);
	printk("flags: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x08);
	printk("NominalAvailableCapacity: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x0a);
	printk("FullAvailableCapacity: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x0c);
	printk("RemainingCapacity: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x0e);
	printk("FullChargeCapacity: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x10);
	printk("AverageCurrent: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x12);
	printk("StandbyCurrent: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x14);
	printk("MaxLoadCurrent: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x16);
	printk("AvailableEnergy: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x18);
	printk("AveragePower: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x1c);
	printk("StateOfCharge: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x1e);
	printk("IntTemperature: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x20);
	printk("StateofHealth: 0x%x\n",ret);
	ret=i2c_smbus_read_word_data(client,0x3c);
	printk("DesignCapacity: 0x%x\n",ret);

	ret=i2c_smbus_write_word_data(client, 0x00, 0x0000);
	if(ret!=0)
		return ret;
	ret=i2c_smbus_read_word_data(client,0x00);
	printk("control status: 0x%x\n",ret);
	return 0;
}

static void bq27410_work(struct work_struct *work)
{
	struct bq27x00_device_info *di;

	di = container_of(work, struct bq27x00_device_info, work.work);
	power_supply_changed(&di->bat);
	schedule_delayed_work(&di->work, BQ27410_DELAY);
}

static int bq27x00_battery_probe(struct i2c_client *client,
				 const struct i2c_device_id *id)
{
	char *name;
	struct bq27x00_device_info *di;
	struct bq27x00_access_methods *bus;
	int num;
	int retval = 0;

	/* Get new ID for the new battery device */
	retval = idr_pre_get(&battery_id, GFP_KERNEL);
	if (retval == 0)
		return -ENOMEM;
	mutex_lock(&battery_mutex);
	retval = idr_get_new(&battery_id, client, &num);
	mutex_unlock(&battery_mutex);
	if (retval < 0)
		return retval;

	name = kasprintf(GFP_KERNEL, "%s-%d", id->name, num);
	if (!name) {
		dev_err(&client->dev, "failed to allocate device name\n");
		retval = -ENOMEM;
		goto batt_failed_1;
	}

	di = kzalloc(sizeof(*di), GFP_KERNEL);
	if (!di) {
		dev_err(&client->dev, "failed to allocate device info data\n");
		retval = -ENOMEM;
		goto batt_failed_2;
	}
	di->id = num;
	di->chip = id->driver_data;

	bus = kzalloc(sizeof(*bus), GFP_KERNEL);
	if (!bus) {
		dev_err(&client->dev, "failed to allocate access method "
					"data\n");
		retval = -ENOMEM;
		goto batt_failed_3;
	}

	i2c_set_clientdata(client, di);
	di->dev = &client->dev;
	di->bat.name = name;
	bus->read = &bq27x00_read_i2c;
	di->bus = bus;
	di->client = client;

	bq27x00_powersupply_init(di);

	bq27410_battery_config(di);

	retval = power_supply_register(&client->dev, &di->bat);
	if (retval) {
		dev_err(&client->dev, "failed to register battery\n");
		goto batt_failed_4;
	}

	INIT_DELAYED_WORK_DEFERRABLE(&di->work, bq27410_work);
	schedule_delayed_work(&di->work, BQ27410_DELAY);

	dev_info(&client->dev, "support ver. %s enabled\n", DRIVER_VERSION);

	return 0;

batt_failed_4:
	kfree(bus);
batt_failed_3:
	kfree(di);
batt_failed_2:
	kfree(name);
batt_failed_1:
	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, num);
	mutex_unlock(&battery_mutex);

	return retval;
}

static int bq27x00_battery_remove(struct i2c_client *client)
{
	struct bq27x00_device_info *di = i2c_get_clientdata(client);

	power_supply_unregister(&di->bat);

	kfree(di->bus);
	kfree(di->bat.name);

	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, di->id);
	mutex_unlock(&battery_mutex);

	kfree(di);

	return 0;
}

/*
 * Module stuff
 */

static const struct i2c_device_id bq27x00_id[] = {
	{ "bq27200", BQ27000 },	/* bq27200 is same as bq27000, but with i2c */
	{ "bq27500", BQ27500 },
	{},
};

static struct i2c_driver bq27x00_battery_driver = {
	.driver = {
		.name = "bq27x00-battery",
	},
	.probe = bq27x00_battery_probe,
	.remove = bq27x00_battery_remove,
	.id_table = bq27x00_id,
};

static int __init bq27x00_battery_init(void)
{
	int ret;

	ret = i2c_add_driver(&bq27x00_battery_driver);
	if (ret)
		printk(KERN_ERR "Unable to register BQ27x00 driver\n");

	return ret;
}
module_init(bq27x00_battery_init);

static void __exit bq27x00_battery_exit(void)
{
	i2c_del_driver(&bq27x00_battery_driver);
}
module_exit(bq27x00_battery_exit);

MODULE_AUTHOR("Rodolfo Giometti <giometti@linux.it>");
MODULE_DESCRIPTION("BQ27x00 battery monitor driver");
MODULE_LICENSE("GPL");
