﻿/**
 * @fileOverview
 */

/**
 * @static
 * @class NVR Core Services.
 * @constructor
 * @description
 * <p>Network Video Recorder Core Services.</p>
 *
 * <p>aui.nvr.core 為一靜態結構，該類別無法被初始化。aui.nvr.core 
 * 提供對NVR伺服器存取所需的各種服務，包含讀寫網路影像裝置設定、
 * 處理使用者資訊、提供當地化語言服務等。</p>
 *
 * <p>當成功連線到NVR伺服器後，aui.nvr.core會將伺服器上的群組設定
 * 及影像裝置設定，下載到本機記憶體中。開發人員可透過
 * aui.nvr.core.RootGroup成員取得所有群組的跟節點。並可從其子結點
 * 中取得各子群組及影像裝置資訊。</p>
 *
 * <p>請利用instanceof關鍵字來判斷子結點為群組節點(aui.nvr.Group)
 * 或影像裝置節點(aui.nvr.NetworkDevice)。如需更進一步區分不同的
 * 影像裝置，則可利用aui.nvr.Camera、aui.nvr.VideoServer、aui.nvr.NVR
 * 等型別來做進一步的區分。</p>
 *
 * <strong>群組節點的階層關係</strong>
 *
 * <p>有關群組及裝置型別的物件階層關係，請參考下圖：</p>
 *
 * <p><img src="images/nvr_model.png"/></p>
 *
 * <p>如圖所示，所有群組項目皆繼承自aui.nvr.GroupItem，而
 * aui.nvr.GroupItem則更進一步區分為aui.nvr.Group及
 * aui.nvr.NetworkDevice兩種。aui.nvr.Group係根節點，其中包含一集合
 * aui.nvr.Group.Items，可包括零到多個aui.nvr.GroupItem子結點，而
 * aui.nvr.NetworkDevice則為葉節點，目前已知的明確葉節點實作有
 * aui.nvr.Camera, aui.nvr.VideoServer, aui.nvr.NVR等三種。</p>
 *
 * <strong>aui.nvr.core之結構。</strong>
 *
 * <p>有關主要服務之結構，請參考下圖：</p>
 *
 * <p><img src="images/nvr_structure.png"/></p>
 *
 * <p>如圖所示，您可以從aui.nvr.core.RootGroup成員取得所有
 * 群組以及影像裝置的清單。</p>
 *
 * <strong>Sample: List all nodes.</strong>
 *
 * @example
 *     function writeItems(item, prefix){
 *         // print the Name member of GroupItem
 *         document.write(
 *             prefix + 
 *             ((item instanceof aui.nvr.Group)?"+(Group) ":"-(Device) ") + 
 *             item.Name + 
 *             "&lt;/br&gt;"
 *         );
 * 
 *         if (item instanceof aui.nvr.Group)
 *         {
 *             prefix += "----";
 *             for(var i = 0 ; i < item.Items.getCount() ; i++)
 *             {
 *                 // access child nodes contained in Items.
 *                 writeItems(item.Items.getItem(i), prefix);
 *             }
 *         }
 *    }
 *    
 *    (function(){
 *          writeItems(aui.nvr.core.RootGroup, "");
 *    })();
 * @property {string} UserAccount 
 * @property {string} Password Password for login NVR 
 * services.
 * @property {string} NetworkAddress Network address of 
 * NVR host, could be IPv4 / IPv6 Address or Domain Name.
 * @property {string} Language Get / Set system UI 
 * language.
 * @property {aui.nvr.GroupItem} RootGroup Root item of 
 * all NetworkDevice. [Note] Changing the root item 
 * directly is not suggested. We recommand client 
 * programmers to use the clear, add & remove function 
 * provided by RootGroup.Items to modify group items.
 */
aui.nvr.core = {};

/** @lends aui.nvr.core.prototype */
aui.nvr.core.pvtCurrentUser = null;
aui.nvr.core.pvtCurrentDataService = null;
//aui.nvr.core.pvtCurrentRecorderDataService = null;
aui.nvr.core.pvtCurrentDevice = null;
aui.nvr.core.pvtCurrentDeviceID = null;
aui.nvr.core.pvtCurrentControl = null;
aui.nvr.core.pvtCurrentVolume = null;
aui.nvr.core.pvtAudioBroadcastStatus = false;
aui.nvr.core.pvtManualRecordAllStatus = false;
aui.nvr.core.pvtCurrentGroup = null;
aui.nvr.core.pvtCurrentCoreGroup = null;
aui.nvr.core.pvtLastView = null;
//aui.nvr.core.pvtUsers = null;
aui.nvr.core.pvtCurrentLanguage = null; // need default as en-us ?
aui.nvr.core.pvtServer = null;
aui.nvr.core.pvtLastViewUsingStatus = false; //last view flag
aui.nvr.core.pvtWaitingTriggerList = [];
aui.nvr.core.pvtInstantPlayback = false;    //Instant Playback status
aui.nvr.core.pvtPatrolStatus = false;       //alldevice patrol / view patrol status
aui.nvr.core.pvtStretchVideo = true;        //StretchToFit status
aui.nvr.core.pvtStretchWindow = true;       //Stretch Window(Canvas) status
aui.nvr.core.pvtEnableDaylight = true;        //Enable Daylight Saving status
aui.nvr.core.pvtLastDevice = null;        //Last Selected Device in Live/Device/Map
aui.nvr.core.pvtDisplayTitleBar = true;        //DisplayTitleBar status
aui.nvr.core.pvtImageRandomNumber = 0;        //Random number for map image to avoid cahce
aui.nvr.core.pvtLastFilteredSortType = null;
aui.nvr.core.pvtLastDeviceKeyword = "";     //Last filtered device keyword in Live/Playback/Setup for CMS2 SourceTree
aui.nvr.core.pvtLastExpandedSources = [true, []];   //Last expanded source node id in Live/Playback for CMS2 SourceTree
aui.nvr.core.pvtMaxLicenseCount = 64;
aui.nvr.core.pvtMaxLicenseCountBySource = 64;

/**
 * Get server instance in used.
 * @function
 * @return {aui.nvr.NetworkDevice}
 * Return server used currently.
 */
aui.nvr.core.getServer = function(){
    return aui.nvr.core.pvtServer;
};

/**
 * Get root group of current Server.
 * @function
 * @return {aui.nvr.GroupItem}
 * Return root group of current server.
 */
aui.nvr.core.getRootGroup = function() {
    return aui.nvr.core.pvtServer.DeviceManager.RootGroup;
};

/**
 * Get all devices connected to current Server.
 * @function
 * @return {aui.collections.ArrayList}
 * Return all devices in current server.
 */
aui.nvr.core.getDevices = function() {
    return aui.nvr.core.pvtServer.DeviceManager.AllDevices;
};

/**
 * Get all users in current Server.
 * @function
 * @return {aui.collections.ArrayList}
 * Return all users in current server.
 */
aui.nvr.core.getUsers = function(){
    return aui.nvr.core.pvtUsers;
};

/**
 * Get user logged in currently.
 * @function
 * @return {aui.nvr.User}
 */
aui.nvr.core.getCurrentUser = function(){
    return this.pvtCurrentUser;
};

/**
 * Get the Group & Device data service in used.
 * @function
 * @return {aui.nvr.data.DeviceDataService}
 * Return the data service in used.
 */
aui.nvr.core.getCurrentDataService = function(){
    return aui.nvr.core.pvtCurrentDataService;
};

/**
 * Set a new data service for operating Group & Device data.
 * @function
 * @param {aui.nvr.data.DeviceDataService} value
 * The instance of new data service.
 */
aui.nvr.core.setCurrentDataService = function(value){
    aui.nvr.core.pvtCurrentDataService = value;
};
    
/**
 * Get focused device.
 * @function
 * @return {aui.nvr.NetworkDevice}
 * Return focused device. The device could be Camera,
 * Video Server, NVR or any instance extended from
 * {aui.nvr.NetworkDevice}.
 */
aui.nvr.core.getCurrentDevice = function(){
    return aui.nvr.core.pvtCurrentDevice;
};

/**
 * Focus to specify device.
 * @function
 * @param {aui.nvr.NetworkDevice} value
 * New focused device. The device could be Camera,
 * Video Server, NVR or any instance extended from
 * {aui.nvr.NetworkDevice}.
 */
aui.nvr.core.setCurrentDevice =function(device){
//    if(aui.nvr.core.pvtCurrentDevice == device) return;
    
    aui.nvr.core.pvtCurrentDevice = device;
    if(device) aui.nvr.core.pvtCurrentDeviceID = device.ID;
    if (aui.nvr.core.Events &&
        aui.nvr.core.Events.onCurrentDeviceChanged)
        aui.nvr.core.Events.onCurrentDeviceChanged.fire();

    aui.nvr.core.pvtIsSender = true;
    //Client.SendDataToServer("nvr", "device:" + device.pvtId);
};

/**
 * Get focused group.
 * @function
 * @return {aui.nvr.Group}
 * Return focused group.
 */
aui.nvr.core.getCurrentGroup = function(){
    return aui.nvr.core.pvtCurrentGroup;
};

/**
 * Focus to specify group.
 * @function
 * @param {aui.nvr.Group} value
 * New focused group.
 */
aui.nvr.core.setCurrentGroup = function(group, status){
//    if(aui.nvr.core.pvtCurrentGroup == group) return; //marked for S3.0.01.06-006

    aui.nvr.core.pvtCurrentGroup = group;
    
    if (aui.nvr.core.Events && 
        aui.nvr.core.Events.onCurrentGroupChanged)
        aui.nvr.core.Events.onCurrentGroupChanged.fire(status ? status : null);
        
    aui.nvr.core.pvtIsSender = true;
    //Client.SendDataToServer("nvr", "group:" + group.pvtId);
};

/**
 * Get last view.
 * Return last view.
 */
aui.nvr.core.getLastView = function(){
    var obj = aui.nvr.core.pvtLastView;
//    if(obj){
//        obj.Items.each(function(each){
//            if(!aui.nvr.core.getServer().DeviceManager.AllDevices.getItem(each.item.ID)) each.item.ID = "0";
//        });
//    }
    return aui.nvr.core.pvtLastView;
};

/**
 * set Last View, and format == core.
 * set Current Layout Style and Current Device for Playback.
 */
aui.nvr.core.setLastView = function(lastView){
    aui.nvr.core.pvtLastView = lastView;
};

/**
 * Get last view Using status.
 * Return last view using status.
 */
aui.nvr.core.getLastViewUsingStatus = function(){
    return aui.nvr.core.pvtLastViewUsingStatus;
};

/**
 * set Last View for Playback, and format == core.
 * set Current Layout Style and Current Device for Playback.
 */
aui.nvr.core.setLastViewUsingStatus = function(status){
    aui.nvr.core.pvtLastViewUsingStatus = status;
};

/**
 * Get Audio Broadcasting status.
 * Return Audio Broadcasting status.
 */
aui.nvr.core.getAudioBroadcastStatus = function(){
    return aui.nvr.core.pvtAudioBroadcastStatus;
};

/**
 * set Audio Broadcasting status
 */
aui.nvr.core.setAudioBroadcastStatus = function(status){
    aui.nvr.core.pvtAudioBroadcastStatus = status;
};

/**
 * Get Manual Record all status.
 * Return Manual Record all status.
 */
aui.nvr.core.getManualRecordAllStatus = function(){
    return aui.nvr.core.pvtManualRecordAllStatus;
};

/**
 * set Audio Broadcasting status
 */
aui.nvr.core.setManualRecordAllStatus = function(status){
    aui.nvr.core.pvtManualRecordAllStatus = status;
};

/**
 * Get focused group and format == core.
 * @function
 * @return {aui.nvr.Group}
 * Return focused group.
 */
aui.nvr.core.getCurrentCoreGroup = function(){
    return aui.nvr.core.pvtCurrentCoreGroup;
};

/**
 * Focus to specify group and format == core.
 * @function
 * @param {aui.nvr.Group} value
 * New focused group.
 */
aui.nvr.core.setCurrentCoreGroup = function(group){
    aui.nvr.core.pvtCurrentCoreGroup = group;
};

/**
 * Focus to specify control.
 * @function
 * New focused control.
 */
aui.nvr.core.setCurrentControl = function(control){
//    if(aui.nvr.core.pvtCurrentControl == control) return;

    aui.nvr.core.pvtCurrentControl = control;
    
    if (aui.nvr.core.Events && 
        aui.nvr.core.Events.onCurrentControlChanged)
        aui.nvr.core.Events.onCurrentControlChanged.fire();
};

/**
 * Get focused control.
 * @function
 * Return focused control.
 */
aui.nvr.core.getCurrentControl = function(){
    return aui.nvr.core.pvtCurrentControl;
};

/**
 * set current volume.
 * @function
 * current volume.
 */
aui.nvr.core.setCurrentVolume = function(value){
    aui.nvr.core.pvtCurrentVolume = value;
    
    if (aui.nvr.core.Events && 
        aui.nvr.core.Events.onCurrentVolumeChanged)
        aui.nvr.core.Events.onCurrentVolumeChanged.fire();
};

/**
 * Get current volume.
 * @function
 * Return current volume.
 */
aui.nvr.core.getCurrentVolume = function(){
    return aui.nvr.core.pvtCurrentVolume;
};

aui.nvr.core.setWaitingTriggerList = function(list){
    aui.nvr.core.pvtWaitingTriggerList.push(list);
};

aui.nvr.core.getWaitingTriggerList = function(){
    return aui.nvr.core.pvtWaitingTriggerList;
}

/**
 * Get Instant Replay status.
 * Return Instant Replay status.
 */
aui.nvr.core.getInstantPlaybackStatus = function(){
    return aui.nvr.core.pvtInstantPlayback;
};

/**
 * set Instant Playback status for Live.
 * set Instant Playback status for Event Trigger, Event Panel dblclick and Context Menu.
 */
aui.nvr.core.setInstantPlaybackStatus = function(status){
    aui.nvr.core.pvtInstantPlayback = status;
    
    if (aui.nvr.core.Events && 
        aui.nvr.core.Events.onPopupWindow)
        aui.nvr.core.Events.onPopupWindow.fire();
};

/**
 * Get Alldevice or View patrol status.
 * Return patrol status.
 */
aui.nvr.core.getPatrolStatus = function(){
    return aui.nvr.core.pvtPatrolStatus;
};

/**
 * set Alldevice patrol /view patrol status for Live.
 */
aui.nvr.core.setPatrolStatus = function(status){
    if(aui.nvr.core.pvtPatrolStatus == status) return;
    aui.nvr.core.pvtPatrolStatus = status;
    
    if (aui.nvr.core.Events && 
        aui.nvr.core.Events.onPatrolStatusChanged)
        aui.nvr.core.Events.onPatrolStatusChanged.fire();
};

/**
 * set Playback channel and current time from event pop trigger.
 */
 aui.nvr.core.setPlaybackChannel = function(obj){
    if(aui.nvr.core.Events && 
       aui.nvr.core.Events.onAddChanneltoPlayback)
       aui.nvr.core.Events.onAddChanneltoPlayback.fire(obj);
 };
 
/**
 * set StretchToFit mode for global.
 */
 aui.nvr.core.setStretchVideo = function(status){
    aui.nvr.core.pvtStretchVideo = status;
    
    if(aui.nvr.core.Events && 
       aui.nvr.core.Events.onStretchVideoChanged)
       aui.nvr.core.Events.onStretchVideoChanged.fire(); 
 };
 
 /**
 * get StretchToFit mode for global.
 */
 aui.nvr.core.getStretchVideo = function(){
    return aui.nvr.core.pvtStretchVideo;
 };

 /**
 * set StretchToFit mode for global.
 */
 aui.nvr.core.setStretchWindow = function(status){
    aui.nvr.core.pvtStretchWindow = status;
    
    if(aui.nvr.core.Events && 
       aui.nvr.core.Events.onStretchWindowChanged)
       aui.nvr.core.Events.onStretchWindowChanged.fire(); 
 };
 
 /**
 * get Stretch Window(Canvas) mode for global.
 */
 aui.nvr.core.getStretchWindow = function(){
    return aui.nvr.core.pvtStretchWindow;
};

/**
* set StretchToFit mode for global.
*/
aui.nvr.core.setDisplayTitleBar = function (status) {
    aui.nvr.core.pvtDisplayTitleBar = status;

    if (aui.nvr.core.Events &&
       aui.nvr.core.Events.onDisplayTitleBarChanged)
        aui.nvr.core.Events.onDisplayTitleBarChanged.fire();
};

/**
* get StretchToFit mode for global.
*/
aui.nvr.core.getDisplayTitleBar = function () {
    return aui.nvr.core.pvtDisplayTitleBar;
};

/**
 * set Daylight Saving mode for Playback.
 */
 aui.nvr.core.setEnableDaylight = function(status){
    aui.nvr.core.pvtEnableDaylight = status;
    
    if(aui.nvr.core.Events && 
       aui.nvr.core.Events.onEnableDaylightChanged)
       aui.nvr.core.Events.onEnableDaylightChanged.fire(status); 
 };
 
 /**
 * get Daylight Saving mode for Playback.
 */
 aui.nvr.core.getEnableDaylight = function(){
    return aui.nvr.core.pvtEnableDaylight;
 };
 
 /**
 * set last selected device in Live/Playback.
 */
 aui.nvr.core.setLastDevice = function(device){
    aui.nvr.core.pvtLastDevice = device;
 };
 
 /**
 * get last selected device in Live/Playback.
 */
 aui.nvr.core.getLastDevice = function(){
    return aui.nvr.core.pvtLastDevice;
 };
 
  /**
 * set last selected map in Live/Playback.
 */
 aui.nvr.core.setLastMap = function(map){
    aui.nvr.core.pvtLastMap = map;
 };
 
 /**
 * get last selected map in Live/Playback.
 */
 aui.nvr.core.getLastMap = function(){
    return aui.nvr.core.pvtLastMap;
 };
 
 /**
 * set last random number of map image url in Live.
 */
 aui.nvr.core.setImageRandomNumber = function(num){
    aui.nvr.core.pvtImageRandomNumber = num;
 };
 
 /**
 * get last random number of map image url in Live.
 */
 aui.nvr.core.getImageRandomNumber = function(){
    return aui.nvr.core.pvtImageRandomNumber;
 };
 
 /**
 * set last filtered sort type in Live/Playback/Setup for CMS2.
 */
 aui.nvr.core.setLastFilteredSortType = function(value){
    aui.nvr.core.pvtLastFilteredSortType = value;
 };
 
 /**
 * get last filtered sort type in Live/Playback/Setup for CMS2.
 */
 aui.nvr.core.getLastFilteredSortType = function(){
    return aui.nvr.core.pvtLastFilteredSortType;
 };
 
/**
 * set last filtered device keyword in Live/Playback/Setup for CMS2.
 */
 aui.nvr.core.setLastDeviceKeyword = function(value){
    aui.nvr.core.pvtLastDeviceKeyword = value;
 };
 
 /**
 * get last filtered device keyword in Live/Playback/Setup for CMS2.
 */
 aui.nvr.core.getLastDeviceKeyword = function(){
    return aui.nvr.core.pvtLastDeviceKeyword;
 };
 
  /**
 * set last expanded source nodes in Live/Playback for CMS2.
 */
 aui.nvr.core.setLastExpandedSources = function(value){
    aui.nvr.core.pvtLastExpandedSources = value;
 };
 
 /**
 * get last expanded source nodes in Live/Playback for CMS2.
 */
 aui.nvr.core.getLastExpandedSources = function(){
    return aui.nvr.core.pvtLastExpandedSources;
 };
 
 /**
 * set max license count.
 */
 aui.nvr.core.setMaxLicenseCount = function(value){
    aui.nvr.core.pvtMaxLicenseCount = value;
 };

 /**
 * get max license count.
 */
 aui.nvr.core.getMaxLicenseCount = function(){
    return aui.nvr.core.pvtMaxLicenseCount;
 };
  /**
 * set max license count.
 */
 //[Bug 16683, 16684] set MaxLicenseCounts by source.
 aui.nvr.core.setMaxLicenseCountBySource = function(value){
    aui.nvr.core.pvtMaxLicenseCountBySource = value;
 };

 /**
 * get max license count.
 */
 //[Bug 16683, 16684] get MaxLicenseCounts by source.
 aui.nvr.core.getMaxLicenseCountBySource = function(){
    return aui.nvr.core.pvtMaxLicenseCountBySource;
 };

aui.nvr.core.pvtOnMessage = function(id, msg) {
/*
    msg = msg.toString();
    try {
        if (msg.indexOf("group") >= 0) {
            var group = aui.nvr.core.pvtGroups.getItem(
                msg.substr(msg.indexOf(":") + 1))[0];
            if (group && !aui.nvr.core.pvtIsSender) aui.nvr.core.setCurrentGroup(group);
        }
        else {
            var device = aui.nvr.core.pvtDevices.getItem(
                msg.substr(msg.indexOf(":") + 1))[0];
            if (device && !aui.nvr.core.pvtIsSender) aui.nvr.core.setCurrentDevice(device);
        }
        
    }
    catch (e) {
    }
    finally {
        aui.nvr.core.pvtIsSender = false;
    }
    */
};